/* ///////////////////////////////////////////////////////////////////////// */
/*  This is part of the source of the OMAP 5912 heterogeneous dual-core      */
/*  MPEG-4 SP video decoder published in ACM Transactions on Embedded        */
/*  Computing Systems, Vol. X, Issue Y.                                      */
/* ------------------------------------------------------------------------- */
/*  The source code is released under GPL license.                           */
/*                                                                           */
/*  Copyright, 2011                                                          */
/*  Multimedia Embedded Systems Labs                                         */
/*  Dept. of Computer Science                                                */
/*  National Chiao Tung University                                           */
/*  Hsinchu, Taiwan.                                                         */
/* ///////////////////////////////////////////////////////////////////////// */

#include "DSPMMU.h"
#include <stdio.h>

void    MMU_WriteTlbEntry(TLB_ENTRY entry, short EntryNumber);
void    MMU_Reset();
void    MMU_ReleaseReset();
void    MMU_Enable();
void    MMU_Disable();
short   MMU_CheckError();
short   MMU_ReadFaultType();
int     MMU_ReadFaultAddress();
TLB_ENTRY MMU_ReadTlbEntry(short EntryNumber);
void    MMU_DisplayTlbEntry(short entryNumber);
void    MMU_Setup();
void    DSP_MMU_STEPUP(void);

void
MMU_WriteTlbEntry(TLB_ENTRY entry, short EntryNumber)
{

    unsigned int VirtualTag;
    unsigned short CamHighReg;
    unsigned short CamLowReg;
    unsigned short RamHighReg;
    unsigned short RamLowReg;

    // The 14 MSB of the 24-bit DSP virtual address form the virtual tag
    VirtualTag = ((entry.VirtualAddress >> 10) & 0x3FFF);

    // Calculate CAM register content

    // The 2 MSB go into the 2 LSB of CAM_H_REG
    CamHighReg = ((VirtualTag >> 12) & 0x3);

    // The 12 LSB go into the 12 MSB of CAM_L_REG */
    CamLowReg = ((VirtualTag & 0xFFF) << 4);

    // Entry is valid and shall be preserved 
    CamLowReg |= VALID | PRESERVED;

    // Select entry type
    CamLowReg = CamLowReg | entry.Type;

    // Calculate RAM registers content 

    // The 16 MSB of the physical address go into RAM_H_REG 
    RamHighReg = entry.PhysicalAddress >> 16;

    // The next 6 significant bits go into RAM_L_REG
    RamLowReg = entry.PhysicalAddress & 0xFC00;

    // Bits 9 and 8 of RAM_L_REG are access permission bits
    RamLowReg = RamLowReg | entry.AccessPermission;

    /* Write to MMU */
    /* Step 1 - Write CAM_H_REG */
    CAM_H_REG = CamHighReg;

    /* Step 2 - Write CAM_L_REG */
    CAM_L_REG = CamLowReg;

    /* Step 3 - Write RAM_H_REG */
    RAM_H_REG = RamHighReg;

    /* Step 4 - Write RAM_L_REG */
    RAM_L_REG = RamLowReg;

    /* Step 5 - Update current entry and protected entry pointers */
    LOCK_REG = ((EntryNumber << 4) | (EntryNumber << 10));

    /* Step 6 - Set LD_TLB_REG to write TLB */
    LD_TLB_REG = WRITE;

}

TLB_ENTRY
MMU_ReadTlbEntry(short EntryNumber)
{
    TLB_ENTRY entry;

    // Clear victim pointer (Bits 9..4 of LOCK_REG)
    LOCK_REG &= 0xFC0F;

// Set victim pointer to the entry that shall be read
    LOCK_REG |= EntryNumber << 4;

// Set LD_TLB_REG to read TLB entry
    LD_TLB_REG = READ;

// Extract entry parameters 
    entry.VirtualAddress = (READ_CAM_H_REG & 0x3) << 22;
    entry.VirtualAddress += (READ_CAM_L_REG & 0xFFF0) << 6;
    entry.PhysicalAddress = READ_RAM_H_REG << 16;
    entry.PhysicalAddress += READ_RAM_L_REG & 0xFC00;
    entry.Type = READ_CAM_L_REG & 0x3;
    entry.AccessPermission = READ_RAM_L_REG & 0x300;

    return entry;
}

void
MMU_DisplayTlbEntry(short entryNumber)
{
    TLB_ENTRY entry;
    entry = MMU_ReadTlbEntry(entryNumber);
    printf("Entry %d : Virtual Address = 0x%x, Physical Address = 0x%x \n",
           entryNumber, entry.VirtualAddress, entry.PhysicalAddress);
}

short
MMU_CheckError()
{
    short   i;
    int     addr;

    i = MMU_ReadFaultType();
    if (i != 0)
    {
        if (i & 0x1)
            printf("Translation Fault! \n");
        if (i & 0x2)
            printf("TLB Miss! \n");
        if (i & 0x4)
            printf("Permission Fault! \n");
        if (i & 0x8)
            printf("Error during prefetch! \n");
        addr = MMU_ReadFaultAddress();
        printf("Fault address : 0x%x (Byte) / 0x%x (Word) \n", addr, addr >> 1);
    }
    return i;
}

void
MMU_Reset()
{
    // Reset DSP MMU to default parameters
    CNTL_REG = RESET;
}

void
MMU_ReleaseReset()
{
    // Release MMU from Reset
    CNTL_REG = REL_RESET;
}

void
MMU_Enable()
{
    // Enable MMU
    CNTL_REG |= ENABLE;
}

void
MMU_Disable()
{
    CNTL_REG &= DISABLE;
}

short
MMU_ReadFaultType()
{
    return F_ST_REG;
}

int
MMU_ReadFaultAddress()
{
    int     i;
    i = FAULT_AD_H_REG << 16;
    i += FAULT_AD_L_REG;
    return i;
}

void
DSP_MMU_STEPUP()
{
    short   i = 0;
    //MMU_EN_CTRL_REG=1;    
    // Setup two TLB entries
    MMU_Setup();

    // Read back the two TLB Entries
    for (i = 0; i < 4; i++)
        MMU_DisplayTlbEntry(i);

    // Loop and wait for MMU error 
    //while(MMU_CheckError()==0);
}

void
MMU_Setup(void)
{
    TLB_ENTRY entry1, entry2, entry3, entry4;

    entry1.VirtualAddress = 0x400000;
    entry1.PhysicalAddress = 0x11000000;
    entry1.AccessPermission = READ_WRITE;
    entry1.Type = SECTION;

    entry2.VirtualAddress = 0x500000;
    entry2.PhysicalAddress = 0x11100000;
    entry2.AccessPermission = READ_WRITE;
    entry2.Type = SECTION;

    entry3.VirtualAddress = 0x600000;
    entry3.PhysicalAddress = 0x11200000;
    entry3.AccessPermission = READ_WRITE;
    entry3.Type = SECTION;

    entry4.VirtualAddress = 0x700000;
    entry4.PhysicalAddress = 0x11300000;
    entry4.AccessPermission = READ_WRITE;
    entry4.Type = SECTION;

    MMU_Reset();
    MMU_ReleaseReset();
    MMU_WriteTlbEntry(entry1, 0);
    MMU_WriteTlbEntry(entry2, 1);
    MMU_WriteTlbEntry(entry3, 2);
    MMU_WriteTlbEntry(entry4, 3);

    MMU_Enable();

}
